#pragma once

#include <QObject>

class EtherCATSlave;

/*
 * Instances of EtherCATInput and EtherCATOutput do not have to
 * live in the same thread as EtherCATSlave.
 * 
 */


/********** EtherCATInput **********/

class EtherCATInput : public QObject {
	Q_OBJECT

	protected:
		EtherCATSlave* slave;
		int byteIndex;
		QByteArray value;
		
		EtherCATInput (EtherCATSlave* slave, int byteIndex, QByteArray value, QObject* parent = nullptr);
		virtual void reportChange () = 0;
		
	signals:
		// Called by EtherCATSlave
		void threadSafeUpdate (QByteArray value);
		
	private slots:
		void update (QByteArray value);
	
	public:
		virtual int size () = 0;
		virtual ~EtherCATInput () {}
};

class EtherCATInputUInt8 : public EtherCATInput {
	Q_OBJECT

	public:
		EtherCATInputUInt8 (EtherCATSlave* slave, int byteIndex, quint8 initialValue = 0, QObject* parent = nullptr);
		quint8 getValue ();
		int size () { return 1; }
		
	protected:
		void reportChange ();

	signals:
		void changed (quint8 value);
};

class EtherCATInputUInt16 : public EtherCATInput {
	Q_OBJECT

	public:
		EtherCATInputUInt16 (EtherCATSlave* slave, int byteIndex, quint16 initialValue = 0, QObject* parent = nullptr);
		quint16 getValue ();
		int size () { return 2; }
		
	protected:
		void reportChange ();

	signals:
		void changed (quint16 value);
};

class EtherCATInputUInt64 : public EtherCATInput {
	Q_OBJECT

	public:
		EtherCATInputUInt64 (EtherCATSlave* slave, int byteIndex, quint64 initialValue = 0, QObject* parent = nullptr);
		quint64 getValue ();
		int size () { return 8; }
		
	protected:
		void reportChange ();

	signals:
		void changed (quint64 value);
};

class EtherCATInputFloat : public EtherCATInput {
	Q_OBJECT

	public:
		EtherCATInputFloat (EtherCATSlave* slave, int byteIndex, float initialValue = 0, QObject* parent = nullptr);
		float getValue ();
		int size () { return 4; }
		
	protected:
		void reportChange ();

	signals:
		void changed (float value);
};


/********** EtherCATOutput **********/

class EtherCATOutput : public QObject {
	Q_OBJECT
		
	protected:
		EtherCATSlave* slave;
		int byteIndex;
		
		EtherCATOutput (EtherCATSlave* slave, int byteIndex, QObject* parent = nullptr);
};

class EtherCATOutputUInt8 : public EtherCATOutput {
	Q_OBJECT
	
	private:
		quint8 value;
		
	public:
		EtherCATOutputUInt8 (EtherCATSlave* slave, int byteIndex, quint8 initialValue = 0, QObject* parent = nullptr);
		quint8 getValue () {return value;}
	
	public slots:
		void setValue (quint8 value);
		void setBit (int index, bool enabled);
};

class EtherCATOutputUInt16 : public EtherCATOutput {
	Q_OBJECT
	
	private:
		quint16 value;
		
	public:
		EtherCATOutputUInt16 (EtherCATSlave* slave, int byteIndex, quint16 initialValue = 0, QObject* parent = nullptr);
		quint16 getValue () {return value;}
	
	public slots:
		void setValue (quint16 value);
		void setBit (int index, bool enabled);
};

class EtherCATOutputUInt64 : public EtherCATOutput {
	Q_OBJECT
	
	private:
		quint64 value;
		
	public:
		EtherCATOutputUInt64 (EtherCATSlave* slave, int byteIndex, quint64 initialValue = 0, QObject* parent = nullptr);
		quint64 getValue () {return value;}
	
	public slots:
		void setValue (quint64 value);
};

class EtherCATOutputFloat : public EtherCATOutput {
	Q_OBJECT
	
	private:
		float value;
		
	public:
		EtherCATOutputFloat (EtherCATSlave* slave, int byteIndex, float initialValue = 0, QObject* parent = nullptr);
		float getValue () {return value;}
	
	public slots:
		void setValue (float value);
};
